/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: download.c 
 *       Updating Hardware flash memory. Core/Bios-switching.
 * -----------------------------------------------------------------*/

#include <xtypedef.h>
#include <xdownld.h>
#include <xregcons.h>
#include <xio.h>
#include <timeout.h>
#include <xregx10.h>
#include <xpciapi.h>

#define CBX_PRINTF(msg) {if (cbx_printf != NULL) cbx_printf((msg));}

extern bx_handlestype bx_handlearray[];

/*---------------------------------------------------------------------------*
 * BestXCoreSwitch
 *
 * Purpose: Switch card's firmware from bios- to core-mode 
 *       
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXCoreSwitch(bx_handletype handle)
{
  BX_TRY_VARS_NO_PROG;

  BX_HANDLECHECK;

  BX_TRY_BEGIN
  {
    if (!BestXIsCore(handle))
    {
      BEST_TIMER tmrTotal;
      bx_porttype port;
      bx_int32 portnum;
      bx_int32 baud;
      BX_TRY(BestXGetPortFromHandle(handle, &port));
      BX_TRY(BestXGetPortnumberFromHandle(handle, & portnum));
      baud = bx_handlearray[handle].param;

      if(BestXHasFirmware(handle))
      {
        BX_TRY(BestXBasicCommand(handle, CMD_CORE_SWITCH,
          NULL, IN_CORE_SWITCH, NULL, NULL));
      }

      if (port == BX_PORT_USB && BestXHasFaust(handle))
      {
        /* FX2 will disconnect, so following BestXCheckConnection() will return false */
        (void)BestXReleaseConnection(handle);
      }

      BestStartTimeout(60000UL, &tmrTotal);
      /* actually this means more or less infinite timeout. 
       * TODO: rethink this 
       */

      BestXLastErrorParamSet (handle, BX_ERRPAR_1, BX_EERR_CAPI_TIMEOUT);
      BestXLastErrorParamStringSet(handle,"Timeout detected by CAPI");
      while (BX_E_OK == BestXCheckConnection(handle))
      {
        BX_TRY_FAIL(BestIsTimeoutDone(&tmrTotal) ? BX_E_TRANSF_CMD : BX_E_OK);
      }

      
      /* This deletes the dyncaps and the DB !! */
      (void) BestXCloseAdv(handle, BX_CLOSE_RESERVED);

      BestStartTimeout(500UL, &tmrTotal);
  
      do {
        ;
      } while (!BestIsTimeoutDone (&tmrTotal));

      BestXConnectTimeoutSet(handle, 20000);
      
      BX_TRY(BestXOpenAdv(&handle, port, portnum, NULL, BX_OPEN_BIOS_SWITCH,0,NULL));
      /* still no dyncaps and DB here !! */
      BestXConnectTimeoutDefSet(handle);

      if (port == BX_PORT_RS232)
      {
        BX_TRY(BestXRS232BaudRateSet(handle, baud));
      }
    }
    else
    {
      /* Card already in core mode, do nothing */
    }
  }
 
  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXBiosSwitch
 *
 * Purpose: Switch card's firmware from core- to bios-mode,
            including or excluding a HW reset
 *       
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXBiosSwitch(bx_handletype handle,bx_int32 boot)
{
  BX_HANDLECHECK;
  {
    BX_TRY_VARS_NO_PROG;

    BX_TRY_BEGIN
    {
      if (BestXIsCore(handle))
      {
        if (boot)
        {
          /* This is a complete card reset including Mephisto */
          BX_TRY(BestXBoardReset(handle));
        }
        else
        {
          /* This just switches to bios mode, without a Mephisto reset */
          BX_TRY(BestXBoardSoftReset(handle));
        }
      }
      else
      {
        /* Card already in bios mode, do nothing */
      }

    }

    BX_ERRETURN(BX_TRY_RET);
  }
}

/*---------------------------------------------------------------------------*
 * BestXFlashProgPrepare
 *
 * Purpose: Prepares the flash memory for programming.
 *          Must be in Core mode.
 *---------------------------------------------------------------------------*/
bx_errtype BestXFlashProgPrepare(bx_handletype handle, bx_fsectiontype section)
{
  BX_DECLARE_FUNCNAME("BestXFlashProgPrepare");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY_FAIL(BestXIsCore(handle) ? BX_E_OK : BX_E_ONLY_CORE);
    BX_TRY_FCT_PARAM(1, section > BX_FSEC_FRED);
    BX_TRY(BestXBasicBlockWrite(handle, FPROG_PREPARE, (bx_int8ptr) & section, 1));
  }

  return BX_TRY_RET;
}

/*---------------------------------------------------------------------------*
 * BestXHexImageLoad
 *
 * Purpose: Loads the given hex-image to the cards RAM
 *          Must be in core mode.       
 *---------------------------------------------------------------------------*/

bx_errtype BestXHexImageLoad(bx_handletype handle, bx_int8ptr heximag, bx_int32 no_of_bytes)
{
  BX_DECLARE_FUNCNAME("BestXHexImageLoad");
  char   cb [10] = {"#???"};
  int    tot_bytes = no_of_bytes;
  void (*cbx_printf) (bx_charptrtype,...) = bx_handlearray[handle].cbx_printf;

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    BX_TRY_FCT_PARAM (2, no_of_bytes > 0x00200000UL);
    BestXCBWriteStringSet(handle, cb);

    while (no_of_bytes) 
    {
#if 0 /* chris: only available in xintern.c */
      BESTX_SPRINTF(cb, "#%3d",
          (tot_bytes - no_of_bytes) * progress_range / tot_bytes +
          progress_start);
#endif
      if(BestXHasFirmware(handle))
      {
        if (no_of_bytes > 127) 
        {
          BX_TRY(BestXBasicBlockWrite(handle, HEXIMAG_LOAD, heximag, 127));
          heximag += 127;
          no_of_bytes -= 127;
        }

        else 
        {
          BX_TRY(BestXBasicBlockWrite(handle, HEXIMAG_LOAD, heximag, no_of_bytes));
          no_of_bytes = 0;
        }
      }
    }
  }

  return BX_TRY_RET;
}

/*---------------------------------------------------------------------------*
 * BestXHexImageProg
 *
 * Purpose: Programs the board's flash memory with data from board's RAM.
 *          Must be in Core mode.
 *---------------------------------------------------------------------------*/
bx_errtype BestXHexImageProg(bx_handletype handle)
{
  BX_TRY_VARS_NO_PROG;
  bx_int8 progkey = 26;
  void (*cbx_printf) (bx_charptrtype,...) = bx_handlearray[handle].cbx_printf;
  BX_TRY_BEGIN
  {
    if(BestXHasFirmware(handle))
    {
      BX_TRY_FAIL(BestXIsCore(handle) ? BX_E_OK : BX_E_ONLY_CORE);
      CBX_PRINTF("$Programming...");
      BX_TRY(BestXBasicBlockWrite(handle, HEXIMAG_PROG, &progkey, 1));
    }
  }

  return BX_TRY_RET;
}

